/*
 * linux/drivers/staging/dspg/backlight/dp52_bl.c - DP52 backlight control
 *
 * (C) Copyright 2011, 2012, DSP Group
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/fb.h>
#include <linux/backlight.h>
#include <video/dp52_bl.h>
#include <linux/mfd/core.h>
#include <linux/mfd/dp52/core.h>
#include <linux/earlysuspend.h>

#define VDCBPEN  (1 << 15)
#define VDCAMPEN (1 << 15)
#define PWM_EN   (1 << 13)

struct dp52_bl_device {
	struct dp52 *dp52;
	struct backlight_device *bl;
	struct dp52_bl_platform_data *pdata;
	struct early_suspend early_suspend_backlight;
	int dcgaincnt, dcgainbp1, dcgainbp2;
	int pwm_cfg1_reg, pwm_cfg2_reg;
};

struct vcd_gain {
	unsigned int gain;
	unsigned int regval;
};

/* Note we multiply the amplifier gain by 100 to increase precision */
static struct vcd_gain vcd_gain_arr[] = {
	{   100, 0x01 },
	{   125, 0x02 },
	{   150, 0x03 },
	{   200, 0x04 },
	{   250, 0x05 },
	{   300, 0x06 },
	{   400, 0x07 },
	{   500, 0x08 },
	{   600, 0x09 },
	{   800, 0x0a },
	{  1000, 0x0b },
	{  1300, 0x0c },
	{  1700, 0x0d },
	{  2200, 0x0e },
	{  2900, 0x0f },
	{  3800, 0x10 },
	{  5000, 0x11 },
	{  6600, 0x12 },
	{  8600, 0x13 },
	{ 10100, 0x14 },
};

static unsigned int
find_bigger_or_equal_gain(unsigned int gain)
{
	unsigned int i;

	for (i = 0; i < (sizeof(vcd_gain_arr) / sizeof(vcd_gain_arr[0])) ; i++)
		if (vcd_gain_arr[i].gain >= gain * 100)
			return i;

	return i - 1;
}

/*
 * In order to calculate the registers we
 * I_led = 1000 / (gain * attenuation)
 * or (in case on gain is bigger the 120)
 * I_led = 1000 / (gain' * attenuation)
 *
 * Where:
 * gain - VDCxxGAIN field in register DC2GAINCNT
 * attenuation - 0.75 + (i+1)/256;  where i is VDCxxATT field int register DC2GAINCNT
 * gain' is composition of DC2GAINBP1 and DC2GAINBP2 registers in some fixed values
 *
 * Note:
 * When gain' registers are in use the gain registers will be zero.
 */
static int
dp52_bl_dcgaincnt_calc(struct dp52_bl_device *dp52_bl, const unsigned int gain_attn)
{
	unsigned int after_gain = 0, after_gain_i = 0;
	unsigned int attn_val = 0x3f;

	dp52_bl->dcgainbp1 = 0x0;
	dp52_bl->dcgainbp2 = 0x0;

	if (gain_attn > 120) {
		dp52_bl->dcgainbp1 = VDCBPEN  | (0x4<<10) | 0x3ff;
		dp52_bl->dcgainbp2 = VDCAMPEN | (0xf<<10) | 0x000;
		after_gain = 5300;
	} else if (gain_attn > 101) {
		dp52_bl->dcgainbp1 = VDCBPEN  | (0x0<<10) | 0x3ff;
		dp52_bl->dcgainbp2 = VDCAMPEN | (0x7<<10) | 0x001;
		after_gain = 3700;
	} else {
		after_gain_i = find_bigger_or_equal_gain(gain_attn);
		after_gain = vcd_gain_arr[after_gain_i].gain;
	}

	/* Set attentuation to decrease after_gain to the appropriate value
	 * According to the datasheet the attenuation multiplier is
	 * 0.75 + (i+1)/256
	 * where i is VDCxxATT
	 *
	 * Note: we multiply by 256 to avoid using fractions, hence:
	 * 192 + (i+1) = val*100*256 / gain
	 * i = val*100*256 / gain - 192 - 1
	 */
	if (after_gain)
		attn_val = ((gain_attn * 100 * 256) / after_gain) - 192 - 1;

	dp52_bl->dcgaincnt = ((attn_val & 0x3f) << 5) |
	                     (vcd_gain_arr[after_gain_i].regval & 0x1f);

	return 0;
}

static void
dp52_bl_set_backlight_closedloop(struct dp52_bl_device *dp52_bl, int brightness)
{
	unsigned int mA, gain_attn;
	/* Set duty-cycle to 75% and enable DCIN2 feedback (new DP) */
	int duty = 0x1f, feedback = 0x4400;

	/* Disable PWM */
	dp52_clr_bits(dp52_bl->dp52, dp52_bl->pwm_cfg2_reg, PWM_EN);

	dp52_write(dp52_bl->dp52, dp52_bl->pwm_cfg1_reg, feedback | duty);

	/* Set brightness by limiting the current flow from PWMx */
	mA = (dp52_bl->pdata->max_ma * brightness) / dp52_bl->bl->props.max_brightness;
	gain_attn = 1000 / ((mA != 0) ? mA : 1000);

	dp52_bl_dcgaincnt_calc(dp52_bl, gain_attn);

	/* Workaround to eliminate the flicker of the backlight when using
	 * gainbp: 1. shutdown backlight, 2. set registers, 3. start backlight
	 */
	if (dp52_bl->dcgainbp1 & VDCBPEN) /* shutdown backlight */
		dp52_write(dp52_bl->dp52, dp52_bl->pwm_cfg1_reg, feedback);

	dp52_write(dp52_bl->dp52, DP52_AUX_DC2GAINCNT1, dp52_bl->dcgaincnt);
	dp52_write(dp52_bl->dp52, DP52_AUX_DC2GAINBP1,  dp52_bl->dcgainbp1);
	dp52_write(dp52_bl->dp52, DP52_AUX_DC2GAINBP2,  dp52_bl->dcgainbp2);

	if (dp52_bl->dcgainbp1 & VDCBPEN) /* start backlight again */
		dp52_write(dp52_bl->dp52, dp52_bl->pwm_cfg1_reg, feedback | duty);

	if (brightness) /* Enable PWM */
		dp52_set_bits(dp52_bl->dp52, dp52_bl->pwm_cfg2_reg, PWM_EN);
}

static void
dp52_bl_set_backlight_openloop(struct dp52_bl_device *dp52_bl, int brightness)
{
	/* Set brightness by changing the duty-cycle of the pwm */
	dp52_write(dp52_bl->dp52, dp52_bl->pwm_cfg1_reg, brightness);

	if (brightness) /* Enable PWM */
		dp52_set_bits(dp52_bl->dp52, dp52_bl->pwm_cfg2_reg, PWM_EN);
	else /* Disable PWM */
		dp52_clr_bits(dp52_bl->dp52, dp52_bl->pwm_cfg2_reg, PWM_EN);
}

static void
dp52_bl_set_backlight_hiz(struct dp52_bl_device *dp52_bl, int brightness)
{
	/* Set brightness by toggling 'hiz' (output to high Z when disabled) */
	if (brightness)
		dp52_set_bits(dp52_bl->dp52, dp52_bl->pwm_cfg2_reg, 1<<7);
	else
		dp52_clr_bits(dp52_bl->dp52, dp52_bl->pwm_cfg2_reg, 1<<7);
}

static int
dp52_bl_update_status(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;
	struct dp52_bl_device *dp52_bl = dev_get_drvdata(&dev->dev);
	int brightness = props->brightness;

	if ((props->power != FB_BLANK_UNBLANK) ||
	    (props->state & BL_CORE_SUSPENDED))
		brightness = 0;

	if (dp52_bl->pdata->mode == DP52_BL_CLOSED_LOOP)
		dp52_bl_set_backlight_closedloop(dp52_bl, brightness);
	else if (dp52_bl->pdata->mode == DP52_BL_HIZ)
		dp52_bl_set_backlight_hiz(dp52_bl, brightness);
	else
		dp52_bl_set_backlight_openloop(dp52_bl, brightness);

	return 0;
}

static int
dp52_bl_get_brightness(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;

	return props->brightness;
}

static struct backlight_ops dp52_bl_ops = {
	.options	= BL_CORE_SUSPENDRESUME,
	.get_brightness	= dp52_bl_get_brightness,
	.update_status	= dp52_bl_update_status,
};

#ifdef CONFIG_HAS_EARLYSUSPEND
static void
dp52_bl_early_suspend(struct early_suspend *es)
{
	struct dp52_bl_device *dp52_bl = container_of(es,
	                struct dp52_bl_device, early_suspend_backlight);

	dp52_bl->bl->props.state |= BL_CORE_SUSPENDED;
	dp52_bl_update_status(dp52_bl->bl);
}

static void
dp52_bl_late_resume(struct early_suspend *es)
{
	struct dp52_bl_device *dp52_bl = container_of(es,
	                struct dp52_bl_device, early_suspend_backlight);

	dp52_bl->bl->props.state &= ~BL_CORE_SUSPENDED;
	dp52_bl_update_status(dp52_bl->bl);
}
#endif

static int __devinit
dp52_bl_probe(struct platform_device *pdev)
{
	struct dp52 *dp52 = dev_get_drvdata(pdev->dev.parent);
	struct dp52_bl_device *dp52_bl;
	struct dp52_bl_platform_data *pdata;
	struct mfd_cell *cell = pdev->mfd_cell;
	struct backlight_properties props;
	int ret = 0;
	int pwm_cfg;

	if (!cell->mfd_data) {
		dev_err(&pdev->dev, "missing platform data\n");
		return -EINVAL;
	}

	pdata = cell->mfd_data;
	if (pdata->modulo < 2)
		pdata->modulo = 2;
	if (pdata->modulo > 256)
		pdata->modulo = 256;

	if (!pdata->max_ma)
		pdata->max_ma = 60;
	if (pdata->max_ma > 1000) {
		dev_err(&pdev->dev, "invalid maximum backlight current consumption\n");
		return -EINVAL;
	}

	/* For closed loop control, DCIN2/PWM2 is the only option */
	if (pdata->mode == DP52_BL_CLOSED_LOOP)
		pdata->pwm = 2;

	if ((pdata->pwm < 1) || (pdata->pwm > 2)) {
		dev_err(&pdev->dev, "invalid PWM specified\n");
		return -EINVAL;
	}

	dp52_bl = kzalloc(sizeof(struct dp52_bl_device), GFP_KERNEL);
	if (!dp52_bl) {
		dev_err(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}

	dp52_bl->dp52 = dp52;
	dp52_bl->pdata = pdata;
	platform_set_drvdata(pdev, dp52_bl);

	if (dp52_bl->pdata->pwm == 1) {
		dp52_bl->pwm_cfg1_reg = DP52_PWM1_CFG1;
		dp52_bl->pwm_cfg2_reg = DP52_PWM1_CFG2;
	} else {
		dp52_bl->pwm_cfg1_reg = DP52_PWM2_CFG1;
		dp52_bl->pwm_cfg2_reg = DP52_PWM2_CFG2;
	}

	memset(&props, 0, sizeof(struct backlight_properties));
	props.type = BACKLIGHT_PLATFORM;
	props.max_brightness = pdata->modulo - 1;
	props.brightness = pdata->modulo - 1;

	dp52_bl->bl = backlight_device_register("dp52-bl", &pdev->dev,
	                                        dp52_bl, &dp52_bl_ops,
	                                        &props);
	if (IS_ERR(dp52_bl->bl)) {
		dev_err(&pdev->dev, "failed to register backlight device\n");
		ret = PTR_ERR(dp52_bl->bl);
		goto err_bl_register;
	}

	/* Enable 4Mhz as source for the PWM */
	dp52_set_bits(dp52, DP52_CMU_CCR2, 1 << (7 + pdata->pwm));

	if (pdata->mode == DP52_BL_CLOSED_LOOP) {
		/* Enable DCIN2 */
		dp52_set_bits(dp52, DP52_AUX_EN, 1 << 6);
		/* Connect backlight current sense to DCIN2 */
		dp52_set_bits(dp52, DP52_AUX_CFG2, 0x1);
	}

	pwm_cfg = (fls(dp52_bl->pdata->modulo) - 2) << 9;
	pwm_cfg |= dp52_bl->pdata->prescalar & 0x7f;
	dp52_write(dp52_bl->dp52, dp52_bl->pwm_cfg2_reg, pwm_cfg);

	backlight_update_status(dp52_bl->bl);

#ifdef CONFIG_HAS_EARLYSUSPEND
	dp52_bl->early_suspend_backlight.suspend = dp52_bl_early_suspend;
	dp52_bl->early_suspend_backlight.resume = dp52_bl_late_resume;
	dp52_bl->early_suspend_backlight.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN;
#endif
	register_early_suspend(&dp52_bl->early_suspend_backlight);

	dev_info(&pdev->dev, "successfully registered\n");

	return 0;

err_bl_register:
	kfree(dp52_bl);

	return ret;
}

static int __devexit
dp52_bl_remove(struct platform_device *pdev)
{
	struct dp52_bl_device *dp52_bl = platform_get_drvdata(pdev);

	unregister_early_suspend(&dp52_bl->early_suspend_backlight);
	backlight_device_unregister(dp52_bl->bl);
	kfree(dp52_bl);

	return 0;
}

static void
dp52_bl_shutdown(struct platform_device *pdev)
{
	struct dp52_bl_device *dp52_bl = platform_get_drvdata(pdev);

	/* disable PWM before rebooting */
	dp52_clr_bits(dp52_bl->dp52, dp52_bl->pwm_cfg2_reg, (1<<7) | PWM_EN);
}

static struct platform_driver dp52_bl_driver = {
	.probe = dp52_bl_probe,
	.remove = __devexit_p(dp52_bl_remove),
	.shutdown = dp52_bl_shutdown,
	.driver = {
		.name = "dp52-backlight",
		.owner = THIS_MODULE,
	},
};

static int __init
dp52_bl_init(void)
{
	return platform_driver_register(&dp52_bl_driver);
}

static void __exit
dp52_bl_exit(void)
{
	platform_driver_unregister(&dp52_bl_driver);
}

module_init(dp52_bl_init);
module_exit(dp52_bl_exit);

MODULE_AUTHOR("DSP Group");
MODULE_ALIAS("platform:dp52-backlight");
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LCD/Backlight DP52");
