#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <mach/hardware.h>
#include <mach/platform.h>
#include <linux/irq.h>
#include <linux/i2c.h>
#include <linux/i2c/ssd2531-ts.h>
#include <linux/slab.h>
#include <linux/gpio.h>

//#define SSD2531_DEBUG
//#define DIRTY_PANEL_COMPENSATION
#define CONTROLLER_DO_THE_SCALING

#ifdef SSD2531_DEBUG
  #define TSDEBUG(x...) printk(x)
#else
  #define TSDEBUG(x...)
#endif

#define le2be16(x)	(((x)<<8 & 0xff00) | ((x)>>8 & 0x00ff))

#define SSD2531_SYSFS_DEV
#ifdef SSD2531_SYSFS_DEV
static struct class *ssd2531_class;
static u16 g_register;
static u8 g_level;
static int ssd2531_init_i2c_sysfs(struct i2c_client *client);
static int ssd2531_class_init(void);
struct ssd2531_device {
	struct device dev;
};
#endif

static struct workqueue_struct *ssd2531_wq;

static const struct i2c_device_id ssd2531_id[] = {
	{ "ssd2531"  },
	{ }
};
MODULE_DEVICE_TABLE(i2c, ssd2531_id);

// global driver data (i2c client data, panel data, screen data and fingers data)
static struct ssd2531_data g_ssd2531_data;


/***************************/
/* i2c Read/Write wrappers */
/***************************/

s32 ssd2531_read_byte(struct i2c_client *client, u8 reg)
{
	return i2c_smbus_read_byte_data(client, reg);
}

int ssd2531_write_byte(struct i2c_client *client, u8 reg, u8 data)
{
	return i2c_smbus_write_byte_data(client, reg, data);
}

u16 ssd2531_read_word(struct i2c_client *client, u16 reg)
{
	u16 le;

	le = i2c_smbus_read_word_data(client, reg);
	return le2be16(le);
}

int ssd2531_write_word(struct i2c_client *client, u8 reg, u16 data)
{
	u16 be_data;
	be_data = le2be16(data);
	return i2c_smbus_write_word_data(client, reg, be_data);
}

u32 ssd2531_read_double_word(struct i2c_client *client, u16 reg)
{
	u32 val=0;

	i2c_smbus_read_i2c_block_data(client,reg,4,(u8*)&val);

	return val;
}

/************************************************************************************************/
/* Very dirty touch screen errors compensation (values calculated from the touch-test2 utility) */
/************************************************************************************************/

#ifdef DIRTY_PANEL_COMPENSATION

/* Compensation offsets in pixels * 10 */
struct POINT touch_compensation[13][9] = 
	{{{-102,-90},{-10,-76},{-76,-58},{-96,-66},{-130,-90},{-188,-94},{-220,-70},{-220,-82},{120,-88}},
	{{-82,4},{-38,20},{-72,24},{-168,22},{-124,40},{-162,46},{-208,28},{-194,40},{106,70}},
	{{-62,-38},{12,-30},{8,-24},{-110,-48},{-138,-6},{-102,-18},{-192,10},{-204,-16},{72,58}},
	{{-88,-48},{6,-54},{-30,-62},{-98,-66},{-26,-68},{16,-56},{-50,-4},{-114,-64},{100,100}},
	{{-78,-70},{44,-62},{18,-90},{-118,-18},{-52,22},{-34,-68},{-74,-90},{-120,-102},{100,78}},
	{{-100,-96},{50,16},{92,-22},{-16,-42},{-82,14},{-64,4},{-198,-88},{-182,-110},{100,38}},
	{{-102,-82},{42,54},{-32,-30},{4,-8},{-178,-56},{-142,-132},{-132,-104},{-162,-160},{104,-90}},
	{{-100,-46},{72,-30},{-64,-86},{6,-38},{-66,-36},{-190,80},{-124,34},{-112,-36},{112,0}},
	{{-100,-100},{6,-90},{-42,-198},{-12,-140},{-40,-94},{-74,-58},{-148,-10},{-224,16},{98,96}},
	{{-114,-222},{-10,-156},{-10,-166},{-14,-168},{-112,-216},{-102,-110},{-122,-110},{-188,-54},{122,-8}},
	{{-112,-236},{24,-194},{22,-200},{-40,-194},{-32,-180},{-92,-222},{-116,-144},{-66,-130},{122,-98}},
	{{-134,-202},{48,-110},{30,-180},{-56,-128},{-104,-150},{-156,-200},{-28,-194},{14,-210},{146,-86}},
	{{-110,110},{86,138},{214,138},{34,100},{-146,116},{-168,122},{-132,116},{0,110},{130,100}}};


// weights are 0-1000 instead of 0-1
void compensate_touch(struct screen_data *screen, struct POINT *point)
{
	struct POINT tl_comp, tr_comp, bl_comp, br_comp;
	struct POINT prev_index, next_index;
	int top_weight, bottom_weight, left_weight, right_weight, tl_weight, tr_weight, bl_weight, br_weight;

	prev_index.x = point->x / 40;
	next_index.x = point->x / 40 + 1;
	prev_index.y = point->y / 40;
	next_index.y = point->y / 40 + 1;

	top_weight = 1000 - (point->y - prev_index.y*40)*1000 / 40;
	bottom_weight = 1000 - (next_index.y*40 - point->y)*1000 / 40;
	left_weight = 1000 - (point->x - prev_index.x*40)*1000 / 40;
	right_weight = 1000 - (next_index.x*40 - point->x)*1000 / 40;

	tl_weight = top_weight * left_weight;
	tr_weight = top_weight * right_weight;
	bl_weight = bottom_weight * left_weight;
	br_weight = bottom_weight * right_weight;

	tl_comp = touch_compensation[prev_index.y][prev_index.x];
	tr_comp = touch_compensation[prev_index.y][next_index.x];
	bl_comp = touch_compensation[next_index.y][prev_index.x];
	br_comp = touch_compensation[next_index.y][next_index.x];

	TSDEBUG("reported point: %dx%d\n", point->x, point->y);
	TSDEBUG("weights: [%d, %d, %d, %d]\n", tl_weight, tr_weight, bl_weight, br_weight);

	point->x += (tl_comp.x * tl_weight / 10000000) + (tr_comp.x * tr_weight / 10000000) + (bl_comp.x * bl_weight / 10000000) + (br_comp.x * br_weight / 10000000);
	point->y += (tl_comp.y * tl_weight / 10000000) + (tr_comp.y * tr_weight / 10000000) + (bl_comp.y * bl_weight / 10000000) + (br_comp.y * br_weight / 10000000);

	if (point->x < screen->min_x) point->x = 0;
	else if (point->x > screen->max_x) point->x = screen->max_x;
	if (point->y < screen->min_y) point->y = 0;
	else if (point->y > screen->max_y) point->y = screen->max_y;

	TSDEBUG("compensated point: %dx%d\n", point->x, point->y);

	return;
}

void ssd2531_to_screen(struct panel_data *panel, struct screen_data *screen, struct POINT *p)
{
	// controller -> screen coordinate convertion
	p->x = p->x * (screen->min_x + screen->max_x) / (panel->max_x - panel->min_x);
	p->y = p->y * (screen->min_y + screen->max_y) / (panel->max_y - panel->min_y);

	// Compensation
	compensate_touch(screen,p);

	return;
}

#endif

/*******************************************************************************************************/
/* End of very dirty touch screen errors compensation (values calculated from the touch-test2 utility) */
/*******************************************************************************************************/

/* read all fingers status and put in fingers[] (not in use) */
void ssd2531_read_fingers_states(struct i2c_client *client, struct finger_data fingers[MAX_POINTERS])
{
	u32 finger_state;
	u8 weight;
	struct POINT finger;
	int i;

	for (i=0 ; i<MAX_POINTERS ; i++) {
		finger_state = ssd2531_read_double_word(client,SSD2531_REG_FINGER_STATUS_BASE + i);
		weight = (finger_state>>28)&0xf;
		finger.x = (finger_state&0xff) | ((finger_state>>12)&0xf00);
		finger.y = ((finger_state>>8)&0xff) | ((finger_state>>8)&0xf00);
		g_ssd2531_data.fingers[i].pressure = weight;
		if (weight) {
#ifdef DIRTY_PANEL_COMPENSATION
			ssd2531_to_screen(&g_ssd2531_data.platform_data->panel, &g_ssd2531_data.platform_data->screen, &finger);
#endif
			fingers[i].x = finger.x;
			fingers[i].y = finger.y;
			fingers[i].status |= FINGER_STATUS_DOWN;
		}
		else fingers[i].status &= ~(FINGER_STATUS_DOWN);
		fingers[i].status |= FINGER_STATUS_DIRTY_BIT;
	}
}

/* read the next event out of the event stack and updates 'fingers' array */
void ssd2531_read_next_event(struct i2c_client *client, struct finger_data fingers[MAX_POINTERS])
{
	u32 event;
	u8 event_type, finger_flag, finger_num = 0;
	struct POINT finger;

	event = ssd2531_read_double_word(client, SSD2531_REG_EVENT_STACK);
	event_type = (event & 0xf);
	finger_flag = ((event>>4) & 0xf);
	finger.x = ((event&0xf0000000)>>20) | ((event>>8) & 0xff);
	finger.y = ((event&0x0f000000)>>16) | ((event>>16) & 0xff);

	switch (finger_flag) {
	case 1:
		finger_num = 0;
		break;
	case 2:
		finger_num = 1;
		break;
	case 4:
		finger_num = 2;
		break;
	case 8:
		finger_num = 3;
		break;
	}

	// Do not handle
	if (finger_num >= MAX_POINTERS) return;

	switch (event_type) {
	case SSD2531_EVENT_TYPE_FINGER_ENTER:
		TSDEBUG("ssd2531-ts: [%d] enter (%04dx%04d)\n", finger_num, finger.x, finger.y);
#ifdef DIRTY_PANEL_COMPENSATION
		ssd2531_to_screen(&g_ssd2531_data.platform_data->panel, &g_ssd2531_data.platform_data->screen, &finger);
#endif
		fingers[finger_num].x = finger.x;
		fingers[finger_num].y = finger.y;
		fingers[finger_num].pressure = 0xf;
		fingers[finger_num].status |= FINGER_STATUS_DOWN | FINGER_STATUS_DIRTY_BIT | FINGER_STATUS_REPORT_ENTER;
		break;
	case SSD2531_EVENT_TYPE_FINGER_MOVE:
		TSDEBUG("ssd2531-ts: [%d] move (%04dx%04d)\n", finger_num, finger.x, finger.y);
#ifdef DIRTY_PANEL_COMPENSATION
		ssd2531_to_screen(&g_ssd2531_data.platform_data->panel, &g_ssd2531_data.platform_data->screen, &finger);
#endif
		fingers[finger_num].x = finger.x;
		fingers[finger_num].y = finger.y;
		fingers[finger_num].pressure = 0xf;
		fingers[finger_num].status |= FINGER_STATUS_DIRTY_BIT;
		break;
	case SSD2531_EVENT_TYPE_FINGER_LEAVE:
		break;
	/* default:
		printk("ssd2531-ts: unknown event - %d\n", event_type); */
	}
}

/* report fingers status to the input device framework */
void ssd2531_report_events(struct input_dev *input_dev, struct finger_data fingers[MAX_POINTERS])
{
	int i;

	int is_finger = 0;
	for (i=0 ; i < MAX_POINTERS ; i++) is_finger |= fingers[i].status & FINGER_STATUS_DOWN;

	for (i=0 ; i < MAX_POINTERS ; i++) {
		if (fingers[i].status & FINGER_STATUS_DOWN) {
#ifdef CONFIG_TOUCHSCREEN_SSD2531_MT
			// Report multi touch events
			input_report_abs(input_dev, ABS_MT_POSITION_X, fingers[i].x);
			input_report_abs(input_dev, ABS_MT_POSITION_Y, fingers[i].y);
			input_report_abs(input_dev, ABS_MT_TOUCH_MAJOR, fingers[i].pressure);
			input_mt_sync(input_dev);
#endif
			// Report single touch events
			input_report_abs(input_dev, ABS_PRESSURE, fingers[i].pressure);
			input_report_abs(input_dev, ABS_X, fingers[i].x);
			input_report_abs(input_dev, ABS_Y, fingers[i].y);

			if (fingers[i].status & FINGER_STATUS_REPORT_ENTER) {
				// On the first touch event we should report btn_touch
				input_report_key(input_dev, BTN_TOUCH, 1);
				fingers[i].status &= ~(FINGER_STATUS_REPORT_ENTER);
			}
			fingers[i].status &= ~(FINGER_STATUS_DIRTY_BIT);
		}
		else {
			if (fingers[i].status & FINGER_STATUS_DIRTY_BIT) {
				input_report_key(input_dev, BTN_TOUCH, 0);
				fingers[i].status &= ~(FINGER_STATUS_DIRTY_BIT);
			}
#ifdef CONFIG_TOUCHSCREEN_SSD2531_MT
			input_report_abs(input_dev, ABS_MT_TOUCH_MAJOR, 0x00);
			input_mt_sync(input_dev);
#endif
		}
	}

	input_sync(input_dev);
}

static int ssd2531_init_controller(struct i2c_client *client);

static int
ssd2531_reset(struct ssd2531_data *ssd2531_data)
{
	if (ssd2531_data->platform_data->reset_gpio > 0) {
		gpio_direction_output(ssd2531_data->platform_data->reset_gpio, 0);
		msleep(1);
		gpio_direction_output(ssd2531_data->platform_data->reset_gpio, 1);
	}
	return ssd2531_init_controller(ssd2531_data->client);
}

/* read event stack status and act according to it */
void ssd2531_read_events_work_handler(struct work_struct *work)
{
	int event_status;
	int i;

	while (((event_status = ssd2531_read_byte(g_ssd2531_data.client, SSD2531_REG_EVENT_STATUS)) > 0) &&
	        (event_status & SSD2531_EVENT_STATUS_STACK_NOT_EMPTY))
	{
		TSDEBUG("ssd2531-ts: event_status = 0x%x\n", event_status);
		/* Verify the fingers status (we use it here instead of the 'leave' event which is buggy) */
		for (i=0 ; i<MAX_POINTERS ; i++) {
			int p = 1<<i;
			if (!(event_status & p) && (g_ssd2531_data.fingers[i].status & FINGER_STATUS_DOWN) && !(g_ssd2531_data.fingers[i].status & FINGER_STATUS_REPORT_ENTER)) {
				TSDEBUG("ssd2531-ts: [%d] leave\n", i);
				g_ssd2531_data.fingers[i].pressure = 0;
				g_ssd2531_data.fingers[i].status |= FINGER_STATUS_DIRTY_BIT;
				g_ssd2531_data.fingers[i].status &= ~FINGER_STATUS_DOWN;
			}
		}

		/* act acording to event_status */
		if (event_status & SSD2531_EVENT_STATUS_LARGE_OBJECT)
		{
			TSDEBUG("ssd2531-ts: large object, ignoring.\n");
			// clear event-stack and set all fingers up
			ssd2531_write_byte(g_ssd2531_data.client, SSD2531_REG_EVENT_STACK_CLEAR, 0);
			for (i=0 ; i<MAX_POINTERS ; i++) {
				g_ssd2531_data.fingers[i].pressure = 0;
				g_ssd2531_data.fingers[i].status |= FINGER_STATUS_DIRTY_BIT;
				g_ssd2531_data.fingers[i].status &= ~(FINGER_STATUS_DOWN);
			}
		}
		/* else if (event_status & SSD2531_EVENT_STATUS_STACK_OVERFLOW) {
			TSDEBUG("ssd2531-ts: event stack overflow\n");
			// clear event-stack and set all fingers up
			ssd2531_write_byte(g_ssd2531_data.client, SSD2531_REG_EVENT_STACK_CLEAR, 0);
			for (i=0 ; i<MAX_POINTERS ; i++) {
				g_ssd2531_data.fingers[i].pressure = 0;
				g_ssd2531_data.fingers[i].status |= FINGER_STATUS_DIRTY_BIT;
				g_ssd2531_data.fingers[i].status &= ~(FINGER_STATUS_DOWN);
			}
		} */
		else if (event_status & SSD2531_EVENT_STATUS_STACK_NOT_EMPTY) {
			ssd2531_read_next_event(g_ssd2531_data.client, g_ssd2531_data.fingers);
		}
		else {
			TSDEBUG("ssd2531-ts: unknown event_status 0x%x\n",event_status);
		}

	}

	if (event_status < 0) {
		printk(KERN_ERR "ssd2531-ts: I2C communication issue; resetting touchscreen controller...\n");
		ssd2531_reset(&g_ssd2531_data);
	} else {
		ssd2531_report_events(g_ssd2531_data.input, g_ssd2531_data.fingers);
	}

	// Re-enable the interrupt
	enable_irq(g_ssd2531_data.client->irq);
}

DECLARE_WORK(read_events_work, ssd2531_read_events_work_handler);


static irqreturn_t ssd2531_handle_interrupt( int irq, void *data )
{
	/* disable the interrupt */
	disable_irq_nosync( irq );

	/* read the whole event stack (async) and only then re-enable the interrupt */
	queue_work(ssd2531_wq, &read_events_work);

	return IRQ_HANDLED;
}


static int ssd2531_init_controller(struct i2c_client *client)
{
	int dev_id = 0, i;
	struct panel_data *panel = &(g_ssd2531_data.platform_data->panel);

	// Chip wakeup, we dont care about nacks here
	i2c_smbus_write_byte_data(client, SSD2531_REG_SYSTEM_EN, 0x00);
	mdelay(20);

	dev_id = ssd2531_read_word(client, SSD2531_REG_CHIP_ID);
	if (dev_id != 0x2531) {
		printk(KERN_ERR "ssd2531-ts: wrong chip id (0x%x)\n", dev_id);
		return -1;
	}
	printk("ssd2531-ts: chip id = 0x%x\n", dev_id);

	// Enable SELFCAP clock
	ssd2531_write_byte(client, SSD2531_REG_CLOCK_EN, SSD2531_CLOCK_EN_SELFCAP | SSD2531_CLOCK_EN_DSP);

	// Set osc freq
	ssd2531_write_byte(client,0xd4,0x08);

	// DAC setting
	ssd2531_write_byte(client, 0xd7, 0x08);

	// Configure drive electrodes
	ssd2531_write_byte(client, SSD2531_REG_DRIVE_ELECTRODES_NUM, panel->driveline_num-6); 	// drive pins #

	// Configure sense electrodes
	ssd2531_write_byte(client, SSD2531_REG_SENSE_ELECTRODES_NUM, panel->senseline_num-6); 	// sense pins #

	// Configure drive pins sequence
	for (i=0 ; i<panel->driveline_num ; i++) 
	{
		ssd2531_write_byte(client, SSD2531_REG_DRIVE_PIN_SLEW_RATE_0+i, panel->drivepin_slew_rates[i]);
	}

	// Configure sub-frame (sampling rate)
	ssd2531_write_byte(client, SSD2531_REG_SUB_FRAMES, 0x03);

	// Configure RAM
	ssd2531_write_byte(client, 0x8d, 0x01);		// enable manual RAM control
	ssd2531_write_byte(client, 0x8e, 0x02);		// RAM bank to scaling RAM
	ssd2531_write_word(client, 0x94, 0x0000);	// initial scaling RAM
	ssd2531_write_byte(client, 0x8d, 0x00);		// disable manual RAM control

	// set scan mode
	ssd2531_write_byte(client, SSD2531_REG_OP_MODE_WRITE, SSD2531_OP_MODE_FAST_166HZ);
	mdelay(100);

	// set charge pump
	ssd2531_write_byte(client, SSD2531_REG_CHARGE_PUMP, SSD2531_CHARGE_PUMP_x6);

	// set driving voltage
	ssd2531_write_byte(client, SSD2531_REG_DRIVE_VOLTAGE, SSD2531_DRIVE_VOLTAGE_LEVEL_15_5V);
	mdelay(300);

	// enable sense filter
	ssd2531_write_byte(client, SSD2531_REG_SENSE_FILTER_EN, 0x01);

	ssd2531_write_byte(client, SSD2531_REG_ENABLE_MOVE_TOLERANCE, 0x01); 	// enable move tolerance
	ssd2531_write_byte(client, SSD2531_REG_MOVE_TOLERANCE_WINDOW, 0x05); 	// set move tolerance
	ssd2531_write_byte(client, SSD2531_REG_MAX_MISSED_FRAMES, 0x00);		// set max miss frames
	ssd2531_write_byte(client, SSD2531_REG_MEDIAN_FILTER, 0x02);		// set median filter
	ssd2531_write_byte(client, SSD2531_REG_FILTER_TYPE, 0x01);		// 2D filter
	ssd2531_write_byte(client, SSD2531_REG_DELTA_DATA_RANGE, 0x00);		// delta data range
	ssd2531_write_byte(client, SSD2531_REG_MIN_FINGER_AREA, 0x01);		// original value was 0x01
	ssd2531_write_byte(client, SSD2531_REG_MIN_FINGER_LEVEL, 0x50);		// original value was 0x50
	ssd2531_write_word(client, SSD2531_REG_MIN_FINGER_WEIGHT, 0x0002);
	ssd2531_write_byte(client, SSD2531_REG_MAX_FINGER_AREA, 0x1e);		// original value was 0x1e

	ssd2531_write_byte(client, SSD2531_REG_IMAGE_SEGMENTATION_DEPTH, 0x03);

	ssd2531_write_byte(client, SSD2531_REG_CG_CALC_MODE, SSD2531_CALC_MODE_WEIGHTED_AVG);
	ssd2531_write_byte(client, SSD2531_REG_ENABLE_MOVING_AVERAGE, 0x01);		// enable finger 1&2 moving average

	// Configure click timing
	ssd2531_write_word(client, SSD2531_REG_SINGLE_CLICK_TIMING, 0x0000);		// single click timing
	ssd2531_write_word(client, SSD2531_REG_DOUBLE_CLICK_TIMING, 0x0000);		// double click timing

	ssd2531_write_byte(client, SSD2531_REG_CG_TOLERANCE, 0x08);			// CG tolerance (was 0x8)
	ssd2531_write_byte(client, SSD2531_REG_X_TRACK_TOLERANCE, 0x14);			// X tracking
	ssd2531_write_byte(client, SSD2531_REG_Y_TRACK_TOLERANCE, 0x14);			// Y tracking
	//ssd2531_write_byte(client, SSD2531_REG_WEIGHT_SCALE_FACTOR, 0x3);

#ifdef CONTROLLER_DO_THE_SCALING
	// let the controller do the panel->screen projection
	ssd2531_write_byte(client, SSD2531_REG_X_SCALE_FACTOR, panel->x_scale);
	ssd2531_write_byte(client, SSD2531_REG_Y_SCALE_FACTOR, panel->y_scale);

	ssd2531_write_byte(client, SSD2531_REG_COORDINATE_REMAP, panel->remap);
#endif

	// Reset init reference
	ssd2531_write_byte(client, SSD2531_RESET_INIT_REFERENCE_PROCEDURE, 0x00);
	ssd2531_write_word(client, SSD2531_REG_EVENT_MASK, 0x801f);

	ssd2531_write_byte(client, SSD2531_REG_IRQ_MASK, 0x0f);			// mask all finger irqs

	// Clear event stack
	ssd2531_write_byte(client, SSD2531_REG_EVENT_STACK_CLEAR, 0x00);

	return 0;
}

static void _ssd2531_suspend(struct ssd2531_data *ssd2531)
{
	disable_irq(ssd2531->client->irq);
	flush_workqueue(ssd2531_wq);
	i2c_smbus_write_byte_data(ssd2531->client, SSD2531_REG_SYSTEM_DISABLE, 0x00);
}

static void _ssd2531_resume(struct ssd2531_data *ssd2531)
{
	i2c_smbus_write_byte_data(ssd2531->client, SSD2531_REG_SYSTEM_EN, 0x00);
	mdelay(20);
	enable_irq(ssd2531->client->irq);
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void ssd2531_early_suspend(struct early_suspend *es)
{
	struct ssd2531_data *ssd2531 =
		container_of(es, struct ssd2531_data, early_suspend);

	_ssd2531_suspend(ssd2531);
}

static void ssd2531_late_resume(struct early_suspend *es)
{
	struct ssd2531_data *ssd2531 =
		container_of(es, struct ssd2531_data, early_suspend);

	_ssd2531_resume(ssd2531);
}
#endif

#if defined(CONFIG_PM) && !defined(CONFIG_HAS_EARLYSUSPEND)
static int ssd2531_suspend(struct i2c_client *client, pm_message_t msg)
{
	_ssd2531_suspend(&g_ssd2531_data);
	return 0;
}

static int ssd2531_resume(struct i2c_client *client)
{
	_ssd2531_resume(&g_ssd2531_data);
	return 0;
}
#else
# define ssd2531_suspend NULL
# define ssd2531_resume NULL
#endif

static int ssd2531_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret;
	struct input_dev *input_dev = NULL;
	struct panel_data *panel;
	struct screen_data *screen;

	g_ssd2531_data.client = client;
	// 10 bits	
	printk("ssd2531-ts: Solomon capacitive touch-screen: address = 0x%02x\n", client->addr);

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_SMBUS_BYTE_DATA | I2C_FUNC_SMBUS_WORD_DATA | I2C_FUNC_SMBUS_I2C_BLOCK)) {
		printk(KERN_ERR "ssd2531-ts: adapter doesn't support i2c interface. aborting!\n");
		return -ENODEV;
	}

	/* Setup platform data (screen and panel data) */
	g_ssd2531_data.platform_data = (struct ssd2531_platform_data*)client->dev.platform_data;

#ifdef SSD2531_SYSFS_DEV
	ret = ssd2531_init_i2c_sysfs(client);
	if (ret < 0) return ret;
#endif

	ssd2531_wq = create_singlethread_workqueue("ssd2531_wq");

	if (g_ssd2531_data.platform_data->reset_gpio > 0) {
		ret = gpio_request(g_ssd2531_data.platform_data->reset_gpio, "ssd2531");
		if (ret < 0) {
			printk(KERN_ERR "ssd2531-ts: cannot request reset gpio. aborting!\n");
			return -ENODEV;
		}
	}

	/* reset the controller and perform the init procedure */
	ret = ssd2531_reset(&g_ssd2531_data);
	if (ret < 0) {
		printk(KERN_ERR "ssd2531-ts: error on init. aborting!\n");
		return -ENODEV;
	}

	screen = &(g_ssd2531_data.platform_data->screen);
	panel = &(g_ssd2531_data.platform_data->panel);

	/* Configure input device */
	input_dev = input_allocate_device();
	if (!input_dev) {
		return -ENOMEM;
	}

	input_dev->name = "ssd2531 multi touch screen";
	set_bit(EV_ABS, input_dev->evbit);
	set_bit(EV_SYN, input_dev->evbit);
	set_bit(EV_KEY, input_dev->evbit);
	set_bit(BTN_TOUCH, input_dev->keybit);

	/* Single Touch-screen events */
#ifdef CONTROLLER_DO_THE_SCALING
	input_set_abs_params(input_dev, ABS_X, screen->min_x - screen->inactive_area_left, screen->max_x + screen->inactive_area_right, 0, 0);
	input_set_abs_params(input_dev, ABS_Y, screen->min_y - screen->inactive_area_top, screen->max_y + screen->inactive_area_bottom, 0, 0);
	input_set_abs_params(input_dev, ABS_PRESSURE, 0, 0xf, 0, 0);
#else
	input_set_abs_params(input_dev, ABS_X, panel->min_x, panel->max_x, 0, 0);
	input_set_abs_params(input_dev, ABS_Y, panel->min_y, panel->max_y, 0, 0);
	input_set_abs_params(input_dev, ABS_PRESSURE, 0, 0xf, 0, 0);
#endif

#ifdef CONFIG_TOUCHSCREEN_SSD2531_MT
	/* Multi Touch-screen events */
  #ifdef CONTROLLER_DO_THE_SCALING
	input_set_abs_params(input_dev, ABS_MT_POSITION_X, screen->min_x - screen->inactive_area_left, screen->max_x + screen->inactive_area_right, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_POSITION_Y, screen->min_y - screen->inactive_area_top, screen->max_y + screen->inactive_area_bottom, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_TOUCH_MAJOR, 0, 0xf, 0, 0);
  #else
	input_set_abs_params(input_dev, ABS_MT_POSITION_X, panel->min_x, panel->max_x, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_POSITION_Y, panel->min_y, panel->max_y, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_TOUCH_MAJOR, 0, 0xf, 0, 0);
  #endif
#endif

	ret = input_register_device(input_dev);
	if (ret) {
		input_free_device(input_dev);
		return -ENODEV;
	}

	g_ssd2531_data.input = input_dev;

	/* Configure controller interrupt */
	ret = request_irq(client->irq, ssd2531_handle_interrupt, IRQ_TYPE_LEVEL_LOW,"ssd2531", &g_ssd2531_data);
	if (ret < 0) {
		printk(KERN_ERR "ssd2531-ts: error on request_irq. aborting!\n");
		input_free_device(input_dev);
		return -ENODEV;
	}

#ifdef CONFIG_HAS_EARLYSUSPEND
	g_ssd2531_data.early_suspend.suspend = ssd2531_early_suspend;
	g_ssd2531_data.early_suspend.resume = ssd2531_late_resume;
	g_ssd2531_data.early_suspend.level = EARLY_SUSPEND_LEVEL_DISABLE_FB;

	register_early_suspend(&g_ssd2531_data.early_suspend);
#endif

	return 0;
}

static int ssd2531_remove(struct i2c_client *client)
{
	flush_workqueue(ssd2531_wq);
	destroy_workqueue(ssd2531_wq);

	free_irq(client->irq, &g_ssd2531_data);
	input_unregister_device(g_ssd2531_data.input);
	input_free_device(g_ssd2531_data.input);

	if (g_ssd2531_data.platform_data->reset_gpio > 0)
		gpio_free(g_ssd2531_data.platform_data->reset_gpio);

    return 0;
}

/******************************************************************************************/

static struct i2c_driver i2c_driver_ssd2531 = {
	.driver = {
		.name = "ssd2531",
	},
	.probe	= ssd2531_probe,
	.remove = ssd2531_remove,
	.suspend = ssd2531_suspend,
	.resume = ssd2531_resume,
	.id_table = ssd2531_id,
};

static int __init ssd2531_init(void)
{
	return i2c_add_driver(&i2c_driver_ssd2531);
}

static void __exit ssd2531_exit(void)
{
	TSDEBUG( "ssd2531-ts: remove\n");

	i2c_del_driver(&i2c_driver_ssd2531);
}

/*********************************/
/* sysfs interface for debugging */
/*********************************/

#ifdef SSD2531_SYSFS_DEV
static int ssd2531_init_i2c_sysfs(struct i2c_client *client)
{
	int ret;
	struct ssd2531_device *new_dev;

	ssd2531_class_init();

	new_dev = kzalloc(sizeof(struct ssd2531_device), GFP_KERNEL);
	if (!new_dev)
		return -ENOMEM;

	new_dev->dev.class = ssd2531_class;

	dev_set_name(&new_dev->dev, "ssd2531");
	dev_set_drvdata(&new_dev->dev, client);

	ret = device_register(&new_dev->dev);
	if (ret) {
		kfree(new_dev);
		return ret;
	}

	return 0;
}


static ssize_t ssd2531_sysfs_read_byte(struct device *dev,
		struct device_attribute *attr,char *buf)
{
	u8 val;
	struct i2c_client *client = g_ssd2531_data.client;
	
	TSDEBUG("read byte from register 0x%x\n", g_register);

	val = ssd2531_read_byte(client, g_register);

	return sprintf(buf, "0x%x\n", val);
}

static ssize_t ssd2531_sysfs_read_word(struct device *dev,
		struct device_attribute *attr,char *buf)
{
	u16 val;
	struct i2c_client *client = g_ssd2531_data.client;
	
	TSDEBUG("read word from register 0x%x\n", g_register);

	val = ssd2531_read_word(client, g_register);

	return sprintf(buf, "0x%x\n", val);
}

static ssize_t ssd2531_sysfs_read_dword(struct device *dev,
		struct device_attribute *attr,char *buf)
{
	u32 val;
	struct i2c_client *client = g_ssd2531_data.client;
	
	TSDEBUG("read double word from register 0x%x\n", g_register);

	val = ssd2531_read_double_word(client, g_register);

	return sprintf(buf, "0x%x\n", val);
}

static ssize_t ssd2531_sysfs_get_register(struct device *dev,
		struct device_attribute *attr,char *buf)
{
	return sprintf(buf, "0x%x\n", g_register);
}

static ssize_t ssd2531_sysfs_store_register(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int rc;
	unsigned long command;

	rc = strict_strtoul(buf, 16, &command);
	if (rc)
		return rc;

	g_register = command;

	return count;
}

static ssize_t ssd2531_sysfs_write_byte(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int rc;
	unsigned long data;
	struct i2c_client *client = g_ssd2531_data.client;

	rc = strict_strtoul(buf, 16, &data);
	if (rc)
		return rc;

	ssd2531_write_byte(client, g_register, (u8)data);

	return count;
}

static ssize_t ssd2531_sysfs_write_word(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int rc;
	unsigned long data;
	struct i2c_client *client = g_ssd2531_data.client;

	rc = strict_strtoul(buf, 16, &data);
	if (rc)
		return rc;

	ssd2531_write_word(client, g_register, (u16)data);

	return count;
}

static ssize_t ssd2531_sysfs_store_level(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int rc;
	unsigned long val;
	struct i2c_client *client = g_ssd2531_data.client;

	rc = strict_strtoul(buf, 16, &val);
	if (rc)
		return rc;

	g_level = (u8)val;

	ssd2531_write_byte(client, SSD2531_REG_MIN_FINGER_LEVEL, g_level);

	return count;
}

static ssize_t ssd2531_sysfs_get_level(struct device *dev,
		struct device_attribute *attr,char *buf)
{
	return sprintf(buf, "0x%x\n", g_level);
}

static struct device_attribute ssd2531_device_attributes[] = {
	__ATTR(read_byte, 0644, ssd2531_sysfs_read_byte, NULL),
	__ATTR(write_byte, 0644, NULL, ssd2531_sysfs_write_byte),
	__ATTR(read_word, 0644, ssd2531_sysfs_read_word, NULL),
	__ATTR(write_word, 0644, NULL, ssd2531_sysfs_write_word),
	__ATTR(read_dword, 0644, ssd2531_sysfs_read_dword, NULL),
	__ATTR(level, 0644, ssd2531_sysfs_get_level, ssd2531_sysfs_store_level),
	__ATTR(register, 0644, ssd2531_sysfs_get_register, ssd2531_sysfs_store_register),
	__ATTR_NULL,
};


static int ssd2531_class_init(void)
{
	ssd2531_class = class_create(THIS_MODULE, "ssd2531");
	if (IS_ERR(ssd2531_class)) {
		printk(KERN_WARNING "Unable to create ssd2531 class; errno = %ld\n",
				PTR_ERR(ssd2531_class));
		return PTR_ERR(ssd2531_class);
	}

	ssd2531_class->dev_attrs = ssd2531_device_attributes;
	return 0;
}

#endif

module_init(ssd2531_init);
module_exit(ssd2531_exit);

MODULE_DESCRIPTION("SSD2531 Capacitive touch-screen driver");
MODULE_LICENSE("GPL");

